from detectors.detectorContext import *
from detectors.fileTools import _getAllFiles, _readFile
from pyparsing import *
from detectors.pyparsingPatterns import *

from metamodels.technologyMetamodel import *
from metamodels.microserviceComponentsMetamodel import *

import os

class Detector(object):
    pass

class SingleFileDetector(Detector):
    def __init__(self):
        self.fileEndings = []
        self.fileNames = []

    def getAllMatchingFiles(self, dir):
        matchingFiles = []
        for file in _getAllFiles(dir):
            for name in self.fileNames:
                if file.endswith(name):
                    matchingFiles.append(file)
            else:
                for ending in self.fileEndings:
                    if file.endswith(ending):
                        matchingFiles.append(file)
        return matchingFiles

    def detect(self, dir, **kwargs):
        evidences = []
        raisedExceptions = []
        print("### DETECT in DIR: " + dir)
        for file in self.getAllMatchingFiles(dir):
            print("*** DETECT in FILE: " + file)
            ctx = DetectorContext(_readFile(file))
            try:
                evidences.extend(self.detectInFile(ctx, file, **kwargs))  
            except DetectorException as de:
                raisedExceptions.append([file, de])
        if evidences == [] and raisedExceptions != []:
            msgs = []
            for f, exception in raisedExceptions:
                msgs.append("in file '" + f + "': " + str(exception))
            raise DetectorException("; ".join(msgs))
        print("return = " + str(evidences))
        return evidences

    # def detectInMatches(self, matches, **kwargs):
    #     evidences = []
    #     raisedExceptions = []
    #     print("### DETECT in MATCHES: " + str(matches))
    #     for match in matches:
    #         print("### DETECT in MATCH: " + match.matchedText)
    #         ctx = DetectorContext(match.matchedText)
    #         try:
    #             newEvidences = self.detectInFile(ctx, match.file, **kwargs)
    #             for e in newEvidences:
    #                 for m in e.matches:
    #                     m.position = match.position + m.position
    #             evidences.extend(newEvidences)  
    #         except DetectorException as de:
    #             raisedExceptions.append([match.file, de])
    #     if evidences == [] and raisedExceptions != []:
    #         msgs = []
    #         for f, exception in raisedExceptions:
    #             msgs.append("in file '" + f + "': " + str(exception))
    #         raise DetectorException("; ".join(msgs))
    #     print("return = " + str(evidences))
    #     return evidences

    def detectInFile(self, ctx, file, **kwargs):
        # to be overridden by subclasses
        return []

def getLinksArgs(**kwargs):
    options = {'source' : None,
        'target' : None}
    options.update(kwargs)
    if options["source"] == None:
        raise Exception("source for link detector is not specified")
    if options["target"] == None:
        raise Exception("target for link detector is not specified")
    return options

# apply multiple detectors across the files
class MultiFileDetector(Detector):
    def __init__(self, detectors):
        super().__init__()
        self.detectors = detectors

    def detect(self, dir, **kwargs):
        evidences = {}
        raisedExceptions = []
        print("### MULTI-DETECT in DIR: " + dir)
        for detector in self.detectors:
            for file in detector.getAllMatchingFiles(dir):
                print("*** DETECT in FILE: " + file)
                ctx = DetectorContext(_readFile(file))
                try:
                    evidences[detector] = detector.detectInFile(ctx, file, **kwargs) 
                except DetectorException as de:
                    raisedExceptions.append([file, de])


        for detector in self.detectors:
            if not detector in evidences.keys() and raisedExceptions != []:
                msgs = []
                for file, exception in raisedExceptions:
                    msgs.append("in file '" + file + "': " + str(exception))
                raise DetectorException("; ".join(msgs))
        # evidences are returned as lists: flatten the lists in the dicts values to a single list
        return [evidence for evidencesList in evidences.values() for evidence in evidencesList]

# def guessNameFromDirectory(file, rootDirectory):
#     fileRelativePath = os.path.relpath(file, rootDirectory)
#     print("file relative path = " + fileRelativePath)
#     fileNameParts = str(fileRelativePath).split("/")
#     print("filenameParts = " + str(fileNameParts))
#     if len(fileNameParts) >= 2:
#         return fileNameParts[0].capitalize()
#     return "" 


def detectJSImport(ctx, file, importName):
    evidences = []
    requireCtx = ctx.matchesPattern(ID + Literal("=") + Literal("require") + 
        Literal("(") + Word("`'\"", max=1) + Literal(importName) + Word("`'\"", max=1) + Literal(")"))
    for match, position in requireCtx.textsAndPositions:
        importVar = match[:match.find("=")].strip()
        print(" +++ import VAR = " + str(importVar))
        evidences.append(ImportEvidence(importName, Match(file, match, position), importVar = importVar))
    return evidences

class JSExpressService(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.fileEndings = ["js"]
      
    def detectInFile(self, ctx, file):
        matches = []
        evidences = []

        imports = detectJSImport(ctx, file, "express")
        for imported in imports:
            matches.extend(imported.matches)
            createAppCtx = ctx.matchesPattern(ID + Literal("=") + Literal(imported.importVar) + 
                Literal("(") + Literal(")"))
            for match, position in createAppCtx.textsAndPositions:
                matches.append(Match(file, match, position))
                appVar = match[:match.find("=")].strip()
                print(" +++ app VAR = " + str(appVar))
                httpHandlerCtx = ctx.matchesPattern(Literal(appVar) + Literal(".") + 
                    oneOf("get put post delete head") + roundBracesBlock + Literal(";"))
                if httpHandlerCtx != None and len(httpHandlerCtx.textsAndPositions) > 0:
                    # print("httpHandlerMatch = " + str(httpHandlerCtx.texts))
                    print(" +++ found http handler, one of: get put post delete")
                    for match, position in httpHandlerCtx.textsAndPositions:
                        matches.append(Match(file, match, position))
                    evidences.append(ServiceEvidence(None, matches, technologyTypes = 
                        ["javascript", "express"]))
        return evidences

# detect use of 'redis - a node.js redis client'
# https://www.npmjs.com/package/redis
class NodeJSRedisAPIClientCalls(SingleFileDetector):
    def __init__(self, redisClientVar):
        super().__init__()
        self.redisClientVar = redisClientVar
        self.fileEndings = ["js"]

    def detectInFile(self, ctx, file, **kwargs):
        matches = []
        # we use only ID here instead of listing all possible redis commands
        # (there are so many, see: https://redis.io/commands)
        useDBOperationCtx = ctx.matchesPattern(Literal(self.redisClientVar) + 
            Literal(".") +  ID + roundBracesBlock + Literal(";"))
        for dbOpMatch, dbOpPosition in useDBOperationCtx.textsAndPositions:
            matches.append(dbOpMatch)
        if len(matches) > 0:
            return [ComponentEvidence(None, matches)]
        return []

class NodeJSRedisDBClient(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.fileEndings = ["js"]

    def detectInFile(self, ctx, file, **kwargs):
        matches = []
        detected = False
        syncAsyncListTypes = []

        imports = detectJSImport(ctx, file, "redis")
        for imported in imports:
            matches.extend(imported.matches)
            try:
                # we detect use of importVar in the original context ctx!
                redisCreateClientCtx = ctx.matchesPattern(ID + Literal("=") + 
                    Literal(imported.importVar) + Literal(".") + Literal("createClient"))
                for ccMatch, ccPosition in redisCreateClientCtx.textsAndPositions:
                    matches.append(ccMatch)
                    redisClientVar = ccMatch[:ccMatch.find("=")].strip()
                    try:
                        syncAsyncRedisCallsDetector = JSCallSyncAsyncDetector(NodeJSRedisAPIClientCalls(redisClientVar))
                        # we detect use of redisClientVar in the original context ctx!
                        callEvidences = syncAsyncRedisCallsDetector.detectInFile(ctx, file, **kwargs)
                        if len(callEvidences) > 0:
                            matches.extend(callEvidences[0].matches)
                            syncAsyncListTypes.extend(callEvidences[0].linkTypes)
                            detected = True
                    except DetectorException: pass
            except DetectorException: pass
        if not detected:
            raise DetectorException("cannot find Redis DB import and use")
        linkTypes = ["resp"]
        syncAsyncListType = _dedupSyncAsyncLinkTypesList(syncAsyncListTypes)
        if syncAsyncListType:
            linkTypes.append(syncAsyncListType)

        return [ComponentEvidence("Redis DB", matches,
                componentTypes = ["redisDB"], linkTypes = linkTypes, technologyTypes = ["redis"])]

class JavaInvocation(SingleFileDetector):
    def __init__(self, invocationName):
        super().__init__()
        self.fileEndings = ["java"]
        self.invocationName = invocationName
    def detectInFile(self, ctx, file, **kwargs):
        evidences = []
        requestCallCtx = ctx.matchesPattern(Literal(self.invocationName) + roundBracesBlock + Literal(";"))
        for match, position in requestCallCtx.textsAndPositions:
            evidences.append(InvocationEvidence(self.invocationName, Match(file, match, position)))
        return evidences
    
class JSInvocation(JavaInvocation):
    def __init__(self, invocationName):
        super().__init__(invocationName)
        self.fileEndings = ["js"]

def containsURLToAlias(string, aliases):
    if "http://" in string:
        string = string[string.find("http://") + 7:] 
    elif "https://" in string:
        string = string[string.find("https://") + 8:]
    else:
        return False
    string = string.lower()
    for alias in aliases:
        if alias.lower() in string:
            return True
    return False

# detects JS request use, not yet considering if it is part of a Promise or not
class JSRequestUse(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.requestDetector = JSInvocation("request")
        self.firstArgPattern = Literal("request") + Literal("(") + OneOrMore(jsString | Word(printables, excludeChars=",);"))
        self.fileEndings = ["js"]

    def detectInFile(self, ctx, file, **kwargs):
        options = getLinksArgs(**kwargs)
        source = options["source"]
        target = options["target"]
        matches = []

        linkEvidences = []
        requestEvidences = self.requestDetector.detectInFile(ctx, file, **kwargs)
        for re in requestEvidences:
            for match in re.matches:
                ctx = DetectorContext(match.matchedText)
                firstArgCtx = ctx.matchesPattern(self.firstArgPattern)
                for argMatch, argPos in firstArgCtx.textsAndPositions:
                    print("checking argMatch = " + str(argMatch))
                    if containsURLToAlias(argMatch, target.aliases):
                        matches.append(match)
        return [LinkEvidence(source, target, matches, linkTypes = 
                    ["restfulHTTP"], technologyTypes = ["javascript", "request"])]

    # def detectInFile(ctx, file, **kwargs):
    #     return self._doDetect(ctx, file, self.requestDetector.detectInFile, **kwargs)
    # def detectInMatches(self, dir, **kwargs):
    #     return self._doDetect(ctx, file, self.requestDetector.detectInMatches, **kwargs)

class JSCallInPromiseLinks(SingleFileDetector):
    def __init__(self, callDetector):
        super().__init__()
        self.promiseDetector = JSInvocation("Promise")
        self.callDetector = callDetector
        self.fileEndings = ["js"]

    def detectInFile(self, ctx, file, **kwargs):
        promiseEvidences = self.promiseDetector.detectInFile(ctx, file, **kwargs)
        promiseMatches = []
        for pe in promiseEvidences:
            promiseMatches.extend(pe.matches)
        callMatches = []
        evidences = []
        for match in promiseMatches:
            try:
                callEvidences = self.callDetector.detectInFile(DetectorContext(match.matchedText), file, **kwargs)
                for ce in callEvidences:
                    callMatches.extend(ce.matches)
                evidences.extend(callEvidences)
            except DetectorException: pass
        if len(evidences) > 0:
            # we know all evidences are of the same type, take the first one and place all matches on it
            evidences[0].matches = callMatches
            return [evidences[0]]
        return []

# detect JS request links and check whether it is an async call via a promise, just a sync call (no promise)
# or a combination of both; other ways to make JS call async need to be added later.
class JSCallSyncAsyncDetector(SingleFileDetector):
    def __init__(self, callDetector):
        super().__init__()
        self.callInPromiseDetector = JSCallInPromiseLinks(callDetector)
        self.callDetector = callDetector
        self.fileEndings = ["js"]

    def detectInFile(self, ctx, file, **kwargs):
        evidencesInPromise = []
        try:
            evidencesInPromise = self.callInPromiseDetector.detectInFile(ctx, file, **kwargs)
        except DetectorException: pass

        evidences = []
        try:
            evidences = self.callDetector.detectInFile(ctx, file, **kwargs)
        except DetectorException: pass

        linkEvidence = None
        linkType = None 
        if len(evidencesInPromise) > 0:
            linkEvidence = evidencesInPromise[0]
            linkType = "asynchronousConnector"
            if len(evidences) > 0:
                if len(evidences[0].matches) > len(linkEvidence.matches):
                    # some more sync links, in addition to the async ones
                    linkType = "syncAsyncConnector"
        elif len(evidences) > 0:
            linkEvidence = evidences[0]
            linkType = "synchronousConnector"
        if linkEvidence == None:
            return []
        linkEvidence.linkTypes.append(linkType)
        return [linkEvidence]

def _dedupSyncAsyncLinkTypesList(syncAsyncListTypes):
    linkType = None
    if "syncAsyncConnector" in syncAsyncListTypes:
        linkType = "syncAsyncConnector"
    elif "synchronousConnector" in syncAsyncListTypes:
        if "asynchronousConnector" in syncAsyncListTypes:
            linkType = "syncAsyncConnector"
        else:
            linkType = "synchronousConnector"
    elif "asynchronousConnector" in syncAsyncListTypes:
        linkType = "asynchronousConnector"
    return linkType

class JSRequestLinks(Detector):
    def __init__(self):
        super().__init__()
        self.syncAsyncRequestDetector = JSCallSyncAsyncDetector(JSRequestUse())
    def detect(self, dir, **kwargs):
        return self.syncAsyncRequestDetector.detect(dir, **kwargs)

def detectGoImport(ctx, file, importString):
    evidences = []
    try:
        importCtx = ctx.matchesPattern("import" + roundBracesBlock).matchesPattern(Optional(ID) +
            Literal("\"") + Literal(importString) + Literal("\""))
        for match, position in importCtx.textsAndPositions:
            if match.strip().startswith("\""):
                evidences.append(ImportEvidence(importString, Match(file, match, position)))
            else:
                alias = match[:match.search("\"")].strip()
                evidences.append(ImportEvidence(importString, Match(file, match, position), alias=alias))
    except DetectorException:
        pass

    try:
        importCtx = ctx.matchesPattern("import" + Optional(ID) + Literal(importString))
        for match, position in importCtx.textsAndPositions:
            if match.strip().startswith(importString):
                evidences.append(ImportEvidence(importString, Match(file, match, position)))
            else:
                alias = match[:match.search(importString)].strip()
                evidences.append(ImportEvidence(importString, Match(file, match, position), alias=alias))
    except DetectorException:
        pass

    if evidences == []:
        raise DetectorException("go import for '" + importString + "' could not be found")
    return evidences

class GoAMQPService(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.fileEndings = ["go"]
      
    def detectInFile(self, ctx, file):
        matches = []
        evidences = []
        detectedImports = detectGoImport(ctx, file, "github.com/streadway/amqp")
        alias = "amqp"
        if len(detectedImports) == 1:
            # there is an amqp import
            matches.extend(detectedImports[0].matches)
            if detectedImports[0].alias != None:
                alias = detectedImports[0].alias
            # check if a connection is established with Dial
            connectionDialCtx = ctx.matchesPattern(Literal(alias) + Literal(".") + Literal("Dial") + roundBracesBlock)
            if len(connectionDialCtx.textsAndPositions) > 0:
                for match, position in connectionDialCtx.textsAndPositions:
                    matches.append(Match(file, match, position))
                evidences.append(ServiceEvidence(None, matches, 
                    technologyTypes = ["go", "amqp"]))
        return evidences

def detectPythonImport(ctx, file, packageName, importName):
    pythonImportAsName = qualifiedID + Optional(Literal("as") + ID)
    pythonImportAsNames = pythonImportAsName + ZeroOrMore(Literal(",") + pythonImportAsName)

    evidences = []
    try:
        importAsNameCtx = ctx.matchesPattern(Literal("import") + Literal(packageName) + 
            Optional(Literal("as") + ID))
        for match, position in importAsNameCtx.textsAndPositions:
            importAsNameCtxElements = match.split()
            if len(importAsNameCtxElements) == 4:
                packageAlias = importAsNameCtxElements[3]
            else:
                packageAlias = None
            evidences.append(ImportEvidence(importName, Match(file, match, position), packageName = packageName, 
                packageAlias = packageAlias, isPackageOnlyImport = True))
    except DetectorException: 
        pass

    try:
        importFromCtx = ctx.matchesPattern(Literal("from") + Literal(packageName) + Literal("import") + 
             (Literal("*") | roundBracesBlock | pythonImportAsNames))
        for match, position in importFromCtx.textsAndPositions:
            importsString = match[match.find("import")+6:].strip()
            alias = None
            if importsString == "*":
                evidences.append(ImportEvidence(importName, Match(file, match, position), packageName = packageName, 
                    alias = alias))
            else:
                if "," in importsString:
                    imports = importsString.split(",")
                else:
                    imports = [importsString]
                for imp in imports:
                    impElts = imp.split()
                    if impElts[0] == importName:
                        if len(impElts) == 3 and impElts[1].strip() == "as":
                            alias = impElts[2]
                        evidences.append(ImportEvidence(importName, Match(file, match, position), 
                            packageName = packageName, alias = alias))
    except DetectorException: 
        pass
    if evidences == []:
        print("raise evidences = "+ str(evidences))
        raise DetectorException("python import for '" + packageName + "." + importName + 
            "' could not be found")
    return evidences  

# determine the possible import strings, e.g. for from f import x, x is possible,
# whereas in import f, then f.x is possible.
def getImportPatternsFromDetectedImports(detectedImports):
        importPatterns = []
        for detectedImport in detectedImports:
            packagePrePattern = None
            if detectedImport.isPackageOnlyImport:
                if detectedImport.packageAlias:
                    packagePrePattern = detectedImport.packageAlias + "."
                else:
                    packagePrePattern = Literal(detectedImport.packageName) + Literal(".")
            if detectedImport.alias:
                importPattern = Literal(detectedImport.alias)
            else:
                importPattern = Literal(detectedImport.name)
            if packagePrePattern:
                importPatterns.append(packagePrePattern + importPattern)
            else: 
                importPatterns.append(importPattern)
        return importPatterns


class PythonFlaskRestfulService(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.fileEndings = ["py"]
      
    def detectInFile(self, ctx, file):
        matches = []
        evidences = []
        detectedImports = detectPythonImport(ctx, file, "flask", "Flask")
        isDetected = False
        print("+++detecImp= " + str(detectedImports))
        if len(detectedImports) > 0:
            for detectedImport in detectedImports:
                matches.extend(detectedImport.matches)
            for importPattern in getImportPatternsFromDetectedImports(detectedImports):
                # match something like: app = Flask(__name__)
                flaskAppCtx = ctx.matchesPattern(ID + Literal("=") + importPattern + roundBracesBlock)
                for match, position in flaskAppCtx.textsAndPositions:
                    matches.append(Match(file, match, position))
                    flaskVarName = match[:match.find("=")].strip()
                    print("###flaskVarName= " + flaskVarName)
                    flaskRouteDefCtx = ctx.matchIndentedPythonBlock(Literal("@" + flaskVarName) + Literal(".") + 
                        Literal("route") + roundBracesBlock + Literal("def") + ID + roundBracesBlock + 
                        Literal(":"), "Flask app route block not found")
                    # we could now check for jsonify and other possible returns in the indented block
                    # but this is hard to impossible to automate, e.g. just returning a dict also creates
                    # JSON and this can be created in many possible ways
                    # => a human needs to inspect there is actually a restful service, and not e.g. 
                    # a web http app.route
                    for match, position in flaskAppCtx.textsAndPositions:
                        matches.append(Match(file, match, position))
                        isDetected = True

        if isDetected:
            evidences.append(ServiceEvidence(None, matches, 
                technologyTypes = ["python", "flask", "restful"]))
        return evidences

class PHPAccessToServerRequestMethod(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.fileEndings = ["php"]

    def detectInFile(self, ctx, file):
        matches = []
        evidences = []
        lineCtx = ctx.matchesPattern(Literal("$") + Literal("_SERVER") + Literal("[") + 
            Word("'\"", max=1) + Literal("REQUEST_METHOD") + Word("'\"", max=1) + Literal("]"))
        for match, position in lineCtx.textsAndPositions:
            evidences.append(Evidence(Match(file, match, position)))
        return evidences

class FunctionInvocation(SingleFileDetector):
    def __init__(self, functionName, fileEndings, insideRoundBracesPattern = None, closingSemicolon = True):
        super().__init__()
        if not isinstance(fileEndings, list):
            fileEndings = [fileEndings]
        self.functionName = functionName
        self.fileEndings = fileEndings
        self.insidePattern = insideRoundBracesPattern
        self.functionPattern = Literal(functionName) + roundBracesBlock 
        if closingSemicolon:
            self.functionPattern = self.functionPattern + Literal(";")

    def detectInFile(self, ctx, file):
        matches = []
        evidences = []
        try:
            lineCtx = ctx.matchesPattern(self.functionPattern)
            for match, position in lineCtx.textsAndPositions:
                if self.insidePattern:
                    try:
                        headerStringCtx = DetectorContext(match).matchesPattern(self.insidePattern)
                        if len(headerStringCtx.textsAndPositions) > 0:
                            matches.append(Match(file, match, position))
                    except DetectorException:
                        pass
                else:
                    matches.append(Match(file, match, position))
        except DetectorException:
            pass

        if len(matches) > 0:
            evidences.append(Evidence(matches))
        else:
            raise DetectorException("invocation to '" + self.functionName + "' not found")
        return evidences


class PHPRestfulService(MultiFileDetector):
    def __init__(self):
        super().__init__([PHPAccessToServerRequestMethod(), 
            # match someting like: header('Content-Type: application/json');
            FunctionInvocation("header", "php", Word("'\"", max=1) + Literal("Content-Type") + 
                Literal(":") + Literal("application/json") + Word("'\"", max=1)),
            # match something like: header('HTTP/1.1[...]');
            FunctionInvocation("header", "php", Word("'\"", max=1) + 
                Literal("HTTP/") + Word(nums, max=1) + Literal(".") + Word(nums, max=1) + 
                Optional(Word(printables, excludeChars="'\"")) + Word("'\"", max=1))
        ])
           
    def detect(self, dir):
        matches = []
        evidences = super().detect(dir)
        if evidences != []:
            for evidence in evidences:
                print("Evidence = " + str(evidence))
                matches.extend(evidence.matches)
            return [ServiceEvidence(None, matches, technologyTypes = ["php", "restful"])]

def detectJavaImport(ctx, file, packageName, importName):
    # import [static] Identifier { . Identifier } [. *] ;
    # javaImportPattern = (Literal("import") + Optional(Literal("static")) + ID + 
    #     ZeroOrMore(Literal(".") + ID) + Optional(Literal(".") + Literal("*")) + Literal(";"))
    importPattern = (Literal("import") + Optional(Literal("static")) + 
        Literal(packageName) + Literal(".") + (Literal("*") | Literal(importName)) + Literal(";"))
    importStaticPattern = (Literal("import") + Literal("static") + 
        Literal(packageName) + Literal(".") + (Literal("*") | Literal(importName)) + Literal(";"))

    evidences = []
    try:
        importCtx = ctx.matchesPattern(importPattern)
        for match, position in importCtx.textsAndPositions:
            evidence = ImportEvidence(importName, Match(file, match, position), 
                packageName = packageName)
            try:
                importStaticCtx = DetectorContext(match).matchesPattern(importStaticPattern)
                if len(importStaticCtx.textsAndPositions) > 0:
                    evidence.isStaticImport = True
            except DetectorException:
                pass
            evidences.append(evidence)
            print("evidences = " + str(evidences))
    except DetectorException: 
        pass
    if evidences == []:
        raise DetectorException("java import for '" + packageName + "." + importName + 
            "' could not be found")
    return evidences  

class JavaSparkService(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.fileEndings = ["java"]
      
    def detectInFile(self, ctx, file):
        matches = []
        evidences = []
        isDetected = False

        packageName = "spark"
        className = "Spark"
        fullyQualifiedName = packageName+ "." + className
        importEvidences = detectJavaImport(ctx, file, packageName, className)
        staticImportDetected = False

        for importEvidence in importEvidences:
            print("Java Import Found")
            matches.extend(importEvidence.matches)
            if importEvidence.isStaticImport:
                staticImportDetected = True

        # if no import evidence is found, still match e.g.: spark.Spark.get(...); 
        classNamePattern = Literal(fullyQualifiedName) + Literal(".")
        if len(importEvidences) > 0:
            # match e.g.: spark.Spark.get(...) and Spark.get(...);
            classNamePattern = (Literal(fullyQualifiedName) | Literal(className)) + Literal(".")
            if staticImportDetected:
                # match example, additionally: get(...);
                classNamePattern = Optional(classNamePattern)

        sparkHTTPMethodCtx = ctx.matchesPattern(classNamePattern + 
            oneOf("get put post delete head") + roundBracesBlock + Literal(";"))
    
        for match, position in sparkHTTPMethodCtx.textsAndPositions:
            matches.append(Match(file, match, position))
            isDetected = True

        if isDetected:
            evidences = [ServiceEvidence(None, matches, 
                technologyTypes = ["java", "spark", "restful"])]
        return evidences


def detectDockerFromStatements(ctx, file, name, version = None):
    if version == None:
        # as version can be "latest" or omitted put into an ARG variable, we don't specify 
        # it in detail here (from can also define an alias [AS <name>]; omitted for now as well)

        # num = Word(nums)
        # versionPattern = num + "." + num + "." + num
        versionPattern = Optional(Literal(":") + Word(printables))
    else:
        # if the user specifies it, it should appear as specified
        versionPattern =  Literal(":") + Literal(version)
    pattern = Literal("FROM") + Literal(name) + versionPattern

    print("DETECT " + str(pattern))

    evidences = []
    try:
        matchedCtx = ctx.matchesPattern(pattern)
        if len(matchedCtx.textsAndPositions) > 0:
            matches = [match for match, position in matchedCtx.textsAndPositions]
            evidences.append(DockerBaseImageEvidence(name, matches))
    except DetectorException: 
        pass
    if evidences == []:
        raise DetectorException("docker FROM statement for base image '" + name + "' not found")
    return evidences  

def detectDockerExposeStatements(ctx, file, port = None):
    if port == None:
        portPattern = Word(nums)
    else:
        portPattern = Literal(port)
        versionPattern =  Literal(":") + Literal(version)
    pattern = Literal("EXPOSE") + portPattern

    evidences = []
    try:
        matchedCtx = ctx.matchesPattern(pattern)
        for match, position in matchedCtx.textsAndPositions:
            port = match[6:].strip()
            evidences.append(DockerExposedPortEvidence(port, match))
    except DetectorException: 
        pass
    if evidences == []:
        if port == None:
            raise DetectorException("docker EXPOSE statement not found")
        else:
            raise DetectorException("docker EXPOSE statement for port '" + str(port) + "' not found")
    return evidences  
     
class DockerEnvStatements(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.fileNames = ["Dockerfile"]

    def detectInFile(self, ctx, file):
        pattern = Literal("ENV") + OneOrMore(Word(printables, excludeChars = "=\\") + 
            Literal("=") + Word(printables, excludeChars = "=\\") + Optional("\\"))
        envLinesPattern = (Word(printables, excludeChars = "=\\") + 
            Literal("=") + Word(printables, excludeChars = "=\\"))

        evidences = []
        try:
            matchedCtx = ctx.matchesPattern(pattern)
            if len(matchedCtx.textsAndPositions) > 0:
                for match, position in matchedCtx.textsAndPositions:
                    envEvidence = DockerEnvEvidence({}, match)
                    envCxt = DetectorContext(match)
                    envLinesCtx = envCxt.matchesPattern(envLinesPattern)
                    for lineMatch, linePosition in envLinesCtx.textsAndPositions:
                        lineParts = lineMatch.split("=")
                        envEvidence.map[lineParts[0].strip()] = lineParts[1].strip()
                    evidences.append(envEvidence)
        except DetectorException: 
            pass
        if evidences == []:
            raise DetectorException("docker ENV statement not found")
        return evidences   

class DockerizedBaseImage(SingleFileDetector):
    def __init__(self, baseImageName, **kwargs):
        super().__init__()
        self.fileNames = ["Dockerfile"]
        self.baseImageName = baseImageName
        options = {'isExposed' : None, 
            'exposedPort' : None,
            'baseImageVersion' : None}
        options.update(kwargs)
        self.isExposed = options['isExposed']
        self.exposedPort = options['exposedPort']
        self.baseImageVersion = options['baseImageVersion']

    def detectInFile(self, ctx, file):
        evidences = []
        isDetected = False

        baseImageEvidences = detectDockerFromStatements(ctx, file, self.baseImageName, self.baseImageVersion)
        for baseImageEvidence in baseImageEvidences:
            evidences.append(baseImageEvidence)
            isDetected = True 
        if isDetected and self.isExposed:
            isDetected = False
            exposeEvidences = detectDockerExposeStatements(ctx, file, self.exposedPort)
            for exposeEvidence in exposeEvidences:
                evidences.append(exposeEvidence)
                isDetected = True 
        if not isDetected:
            evidences = []
        return evidences

class NginxLocations(SingleFileDetector):
    def __init__(self):
        super().__init__()
        self.fileEndings = ["conf", "conf.template"]

    def detectInFile(self, ctx, file, **kwargs):
        serverBlockPattern = Literal("server") + curlyBracesBlock
        locationPattern = Literal("location") + Word(printables, excludeChars="}{") + curlyBracesBlock
    
        evidences = []
        try:
            matchedCtx = ctx.matchesPattern(serverBlockPattern).matchesPattern(locationPattern)
            for match, position in matchedCtx.textsAndPositions:
                evidences.append(Evidence(match))
        except DetectorException: 
            pass
        if evidences == []:
            raise DetectorException("nginx location block not found")
        return evidences

class NginxLocationsToRESTAPIClient(NginxLocations):
    def detectInFile(self, ctx, file, **kwargs):
        # TODO: here caching the result on project might be an option in order not to run NginxLocations 3 times
        locationEvidences = super().detectInFile(ctx, file)
        if len(locationEvidences) > 0:
            return [ComponentEvidence("REST API Client", _getAllMatchesOfEvidenceList(locationEvidences),
                componentTypes = ["client"], linkTypes = ["restfulHTTP"], technologyTypes = ["nginx"])]
        raise DetectorException("could not find nginx locations in file '" + file + "'")

class NginxLocationsToWebUIClient(NginxLocations):
    def detectInFile(self, ctx, file, **kwargs):
        # TODO: here caching the result on project might be an option in order not to run NginxLocations 3 times
        locationEvidences = super().detectInFile(ctx, file)

        rootPattern = Literal("root") + Word(printables, excludeChars=";") + Literal(";")
        matches = []

        for locationEvidence in locationEvidences:
            for locationMatch in locationEvidence.matches:
                try:
                    ctx = DetectorContext(locationMatch).matchesPattern(rootPattern)
                    if len(ctx.textsAndPositions) > 0:
                        matches.append(locationMatch)
                except DetectorException: 
                    pass
        if matches != []:
            return [ComponentEvidence("Web UI Client", matches,
                componentTypes = ["webUI"], linkTypes = ["http"], technologyTypes = ["nginx"])]
        raise DetectorException("could not find nginx locations that contain a web root in file '" + file + "'")

class NginxLocationToServiceLinks(NginxLocations):
    def detectInFile(self, ctx, file, **kwargs):
        options = getLinksArgs(**kwargs)
        source = options["source"] 
        target = options["target"]

        # use identifier to guess whether this is a location pointing to this link
        targetIdentifier = target.identifier



        # possibleLocationParts = [targetIdentifier]

        # possibleHostParts = []

        # # use envVars containing the identifier as well
        # print("envVar = " + str(envVars))
        # for var in envVars:
        #     if targetIdentifier.lower() in var.lower() and "host" in var.lower():
        #         possibleHostParts.append(var)
        #     varValue = envVars[var]
        #     if targetIdentifier.lower() in varValue.lower():
        #         possibleLocationParts.append(varValue)

        locationEvidences = super().detectInFile(ctx, file)

        # print("possible host= " + str(possibleHostParts))
        # print("possible loc= " + str(possibleLocationParts))

        locationStartPattern = Literal("location") + Word(printables, excludeChars="}{")
        matches = []

        for locationEvidence in locationEvidences:
            locationMatches = False
            for locationMatch in locationEvidence.matches:
                locationStartCtx = DetectorContext(locationMatch).matchesPattern(locationStartPattern)
                for match, position in locationStartCtx.textsAndPositions:
                    urlLocation = match[8:].strip()
                    for alias in target.aliases:
                        if alias.lower() in urlLocation.lower():
                            matches.append(match)
                            locationMatches = True
            # if the location does not yet match, match the host pattern
            if not locationMatches:
                for alias in target.aliases:
                    urlPatternEvidences = detectProxyPassInNginxLocation(locationEvidence.matches, alias)
                    print("ev = " + str(urlPatternEvidences))
                    if len(urlPatternEvidences) > 0:
                        matches.extend(locationEvidence.matches)

        if matches == []:
            raise DetectorException("could not find location that matches '" + targetIdentifier + "'")

        return [LinkEvidence(source, target, matches, linkTypes = 
                        ["restfulHTTP"], technologyTypes = ["nginx"])]


def detectProxyPassInNginxLocation(locationMatches, urlPart = None):
    pattern = Literal("proxy_pass") + httpUrlStart + Word(printables, excludeChars=";") + Literal(";")
    evidences = []

    for locationMatch in locationMatches:
        locationCtx = DetectorContext(locationMatch)
        try:
            print("find in " + locationMatch)
            matchedCtx = locationCtx.matchesPattern(pattern)
            for match, position in matchedCtx.textsAndPositions:
                url = match[10:-1] 
                if urlPart != None:
                    if urlPart.lower() in url.lower():
                        evidences.append(NamedEvidence(match, url))
                else:
                    evidences.append(NamedEvidence(match, url))
        except DetectorException: 
            pass

    return evidences


class NginxProxyPass(NginxLocations):
    def detectInFile(self, ctx, file):
        locationEvidences = super().detectInFile(ctx, file)
        evidences = []
        for locationEvidence in locationEvidences:
            try:
                evidences.extend(detectProxyPassInNginxLocation(locationEvidence.matches))
            except DetectorException: 
                pass
        if evidences == []:
            raise DetectorException("nginx location blocks contain no proxy_pass fields")
        return evidences

class NginxAPIGateway(Detector):
    def __init__(self):
        super().__init__()
        self.nginxPresentInDockerDetector = DockerizedBaseImage("nginx", isExposed = True)
        self.proxyPassDetector = NginxProxyPass()

    def detect(self, dir, **kwargs):
        # here an alternative detection for nginx being present but not 
        # dockerized should be added.
        matches = []
        evidencesNginxPresent = self.nginxPresentInDockerDetector.detect(dir, **kwargs)
        if evidencesNginxPresent != []:
            for evidence in evidencesNginxPresent:
                matches.extend(evidence.matches)
            evidencesProxyPass = self.proxyPassDetector.detect(dir, **kwargs)
            if evidencesProxyPass != []:
                for evidence in evidencesProxyPass:
                    matches.extend(evidence.matches)
                return [ServiceEvidence("NGINX API Gateway", matches, componentTypes = ["facade"],
                    technologyTypes = ["nginx"])]
        return []

class Evidence(object):
    def __init__(self, matches):
        if not isinstance(matches, list):
            matches = [matches]
        self.matches = matches

class NamedEvidence(Evidence):
    def __init__(self, name, matches):
        super().__init__(matches)
        self.name = name

class KeyValueMapEvidence(Evidence):
    # map is expected to hold data in a dict of the from {key:value, ...}
    def __init__(self, map, matches, **kwargs):
        super().__init__(matches, **kwargs)   
        self.map = map

class ImportEvidence(NamedEvidence):
    def __init__(self, name, matches, **kwargs):
        options = {'packageName' : None, 
            'alias' : None,
            'packageAlias' : None,
            'isPackageOnlyImport' : False,
            'importVar' : None,
            'isStaticImport': False}
        # set isPackageOnlyImport to true e.g. for "import flask" in python where the
        # package and used Python element is imported
        options.update(kwargs)
        super().__init__(name, matches)   
        self.alias = options["alias"]
        self.packageAlias = options["packageAlias"]
        self.isPackageOnlyImport = options["isPackageOnlyImport"]
        self.importVar = options["importVar"]
        self.isStaticImport = options["isStaticImport"]

class ComponentEvidence(NamedEvidence):
    def __init__(self, name, matches, **kwargs):
        options = {'technologyTypes' : [],
            'componentTypes' : [],
            # possible link types can be collected on the component if its matches might
            # be considered as enough evidence for a link as well (e.g., for a web server
            # offering the component already implies possible links to the web clients)
            'linkTypes' : []}
        options.update(kwargs)
        super().__init__(name, matches)
        self.componentTypes = options["componentTypes"]
        self.technologyTypes = options["technologyTypes"]
        self.linkTypes = options["linkTypes"]

class ServiceEvidence(ComponentEvidence):
    def __init__(self, name, matches, **kwargs):
        super().__init__(name, matches, **kwargs)
        print("***componentTypes= " + str(self.componentTypes))
        self.componentTypes = ["service"] + self.componentTypes
        print("***componentTypes= " + str(self.componentTypes))

class InvocationEvidence(NamedEvidence):
    pass

class LinkEvidence(Evidence):
    def __init__(self, source, target, matches, **kwargs):
        options = {'technologyTypes' : [],
            'linkTypes' : []}
        options.update(kwargs)
        super().__init__(matches)
        self.source = source
        self.target = target
        self.linkTypes = options["linkTypes"]
        self.technologyTypes = options["technologyTypes"]

class DockerBaseImageEvidence(NamedEvidence):
    pass
class DockerExposedPortEvidence(Evidence):
    def __init__(self, port, matches, **kwargs):
        super().__init__(matches, **kwargs)   
        self.port = port
class DockerEnvEvidence(KeyValueMapEvidence):
    pass

def _getAllMatchesOfEvidenceList(evidenceList):
    matches = []
    for evidence in evidenceList:
        matches.extend(evidence.matches)
    return matches

class Match():
    def __init__(self, file, matchedText, position):
        self.file = file
        self.matchedText = matchedText
        self.position = position
    def __str__(self):
        return f"Match: {{File: {self.file}, Pos: {self.position}, Matched Text: {self.matchedText}}}"